%% Analog input, Tachometer, and Counter Measurements
%   
%  Copyright (C) 2010 DataTranslation Inc.
%

%% Introduction
% This example demonstrates how to perform anlaog input,
% tachometer, and counter measurements.


%%
% Open connection to instrument.
% devRsrcName is an IVI logical name or an instrument specific string that 
% identifies the address of the instrument, such as a VISA resource descriptor
% string.
devRsrcName = 'TCPIP::192.43.218.135::INSTR';

% Create device object
deviceObj = icdevice('DT8837_DT8837', devRsrcName);

try
    %% Connect device object to the DT8837 instrument
    connect(deviceObj);
    
    %% Get instrument identity
    
    comobj = get(deviceObj, 'Identity');
    propertyValue = get(comobj, 'InstrumentModel');
    str = strcat ('InstrumentModel= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'InstrumentManufacturer');
    str = strcat ('InstrumentManufacturer= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'InstrumentFirmwareRevision');
    str = strcat ('InstrumentFirmwareRevision= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Description');
    str = strcat ('Description= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Identifier');
    str = strcat ('Identifier= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Vendor');
    str = strcat ('Vendor= ',propertyValue);  
    disp(str);
    propertyValue = get(comobj, 'Revision');
    str = strcat ('Revision= ',propertyValue);  
    disp(str);
    
    %% Display the AnalogInputChannels collection
    analogInputChannelCount = deviceObj.Analoginputchannels.Count;
    
    % Preallocate array.
    analogInputChannelNames={analogInputChannelCount};

    for iLoop= 1:analogInputChannelCount
        analogInputChannelNames{iLoop, 1} = invoke(deviceObj.Analoginputchannels, 'Name', iLoop);
    end
    disp(analogInputChannelNames)
    
    %% Enable the channels to be measured by the DT8837 instrument
    % Note: The counter channels and Tachometer channel are enabled within the
    % analog input channels collection and their data is embedded in the
    % analog input stream.
    
    groupObj = get(deviceObj, 'Analoginputchannels');
    groupObj = groupObj(1);
    
    % Enable analog input channel 1 measurment
    AiChanObject = invoke(groupObj, 'Item', 'AD.1');
    AiChanObject.Enabled = true;
    
    % Enable Counter 1 measurment
    groupObj = get(deviceObj, 'Analoginputchannels');
    groupObj = groupObj(1);
    Ctr1Object = invoke(groupObj, 'Item', 'Ctr.1');
    Ctr1Object.Enabled = true;
    
    % Enable Tachometer measurment   
    TachObject = invoke(groupObj, 'Item', 'Tach.1');
    TachObject.Enabled = true;
    
    %% Configure the analog input subsystem
    
    GroupObj = get(deviceObj, 'Analoginputacquisition');
    analogInputAcquisitionObj = GroupObj(1);
    
    % Disable wrapping, so that data does not get overwritten in the
    % hardware FIFO when the FIFO is full
    set(analogInputAcquisitionObj, 'WrapEnabled', 'off');
  
    % Set the clock frequency
    set(deviceObj.Analoginputacquisition(1), 'SampleRate', 1000);
    
    % Set the trigger source to trigger the instrument immediately 
    set(deviceObj.Analoginputtrigger(1), 'Source', 'IMMEdiate');
    
    %% Configure Counter 1
    
    set(deviceObj.Counter(1), 'StartEdge', 'DT8837CounterEdgeGateRising');
    set(deviceObj.Counter(1), 'StopEdge', 'DT8837CounterEdgeGateFalling');
    
    % Disable the SelfClear flag. When disabled, the previous measurement value
    % from the counter is held and returned on any read of the counter before
    % the next continuous measurement operation is completed.
    set(deviceObj.Counter(1), 'SelfClearEnable', 'off');
    
    %% Configure the tachometer
    
    set(deviceObj.Tach(1), 'TachEdgeType', 'DT8837TachEdgeTypeRising');
    
    % Enable the SelfClear flag. When enabled, clears the latched measurement value
    % from the tachometer after it is read; zeros are returned on any subsequent
    % reads of the tachometer before the next continuous measurement operation is completed. 
    set(deviceObj.Tach(1), 'SelfClearEnabled', 'on');
    
    % Enable the use of the MSB of the value to indicate new or old data.
    % If staleValueEnabled is 'on', the MSB of the value indicates whether the 
    % measurement is fresh or stale. When StaleValueEnabled is 'on', the MSB of
    % the value is set to 0 to indicate fresh data or 1 to indicate stale data.
    % Reading the value before the measurement is complete returns an MSB of 1 to
    % indicate stale data.
    set(deviceObj.Tach(1), 'StaleValueEnabled', 'on');
     
    %% Initiate the measurement and get the data. 
    invoke(analogInputAcquisitionObj, 'Arm');
    invoke(analogInputAcquisitionObj, 'Initiate');
    
   
    isRunning = false;
    ScanIndex = 0;
    RequestedScansToRead = 100;
    
    % Wait for the analog input subsystem to change state to Running and
    % the requested number of scans are read
    while (isRunning == false || ScanIndex < (RequestedScansToRead-1))
    [ScanIndex, isRunning, isArmed, isTriggered, isADSyncDetected, isADFifoOverflow] = invoke(analogInputAcquisitionObj, 'GetStatus', 0, 0, 0, 0, 0, 0);
    end
    
    % Read 100 scans from the instrument
    RequestedScansIndex = 0;
    [ActualScansIndex, ActualScansRead, StartTimeInSeconds, StartTimeInMilliSeconds, samples] = invoke(analogInputAcquisitionObj, 'Fetch', int32(RequestedScansIndex), int32(RequestedScansToRead), int32(0), int32(0), int32(0), int32(0), [0;0]);
    
    RequestedScansIndex = ActualScansIndex+ActualScansRead;
    disp(['Actual Scans Index: ', num2str(ActualScansIndex) ,'  Actual Scans Read: ', num2str(ActualScansRead)]);
    
    % reshape the samples into 3 rows (each row is associated with the
    % channel where the first row is associated with analog input channel 1,
    % the second row is associated with the tachometer channel and the
    % third row is assocated with counter 1 channel
    B = reshape(samples,[3 100]);
    
    % display the data for each channel
    disp(B(1,:))
    disp(B(2,:))
    disp(B(3,:))
    
    %% Stop the acquisition
    invoke(analogInputAcquisitionObj, 'Abort');
    
 
catch DT8837error
    disp(['Error id: ', DT8837error.identifier]);
    disp(['Error Message: ',DT8837error.message]);
end

% Disconnect the device object from the instrument and remove it from memory
disconnect(deviceObj);
delete(deviceObj);   

